"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.addUser = addUser;
exports.getUsers = getUsers;
exports.getUser = getUser;
exports.deleteUser = deleteUser;
exports.updateUser = updateUser;
exports.getAppUsers = getAppUsers;
const common_service_1 = require("../../service/common.service");
const api_response_1 = require("../../api.response");
const logger_service_1 = require("../../service/logger.service");
const utills_1 = require("../../service/utills");
const users_entity_1 = __importDefault(require("../../user/entities/users.entity"));
const app_user_entity_1 = __importDefault(require("../../user/entities/app-user.entity"));
const admin_user_entity_1 = __importDefault(require("../../user/entities/admin-user.entity"));
const mongoose_1 = __importDefault(require("mongoose"));
//==================== ADD USER =================/
function addUser(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a;
        const { firstName, lastName, phone, email, role, status } = req.body;
        // Validate required fields
        if (!email) {
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.allFieldsRequired);
        }
        try {
            // Get user info from token or default to USER type
            const userInfo = (0, common_service_1.getTokenUserInfo)(req);
            // Build the query for existing users based on email, and optionally phone for ADMINs
            const query = {
                $or: [{ email }],
            };
            if ((userInfo === null || userInfo === void 0 ? void 0 : userInfo.userType) === utills_1.USER_TYPE.SUPER_ADMIN) {
                query.$or.push({ phone });
            }
            // Check if user already exists
            const existingUser = yield users_entity_1.default.findOne(query);
            // If an existing user is found
            if (existingUser) {
                if (existingUser.userType === utills_1.USER_TYPE.SUB_ADMIN) {
                    return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.signUp.emailTaken);
                }
                // Handle unverified users (AppUser)
                const existingAppUser = yield app_user_entity_1.default.findOne(query);
                if (existingAppUser && !existingAppUser.isVerify) {
                    const otp = Math.floor(1000 + Math.random() * 9000).toString();
                    existingUser.otp = otp;
                    yield existingUser.save();
                    // const emailSent = await sendEmail({
                    //   userEmail: existingUser.email,
                    //   subject: "Your OTP Code",
                    //   text: `Your OTP code is ${otp}. Please use this code to verify your email.`,
                    //   html: `<p>Your OTP code is <strong>${otp}</strong>. Please use this code to verify your email.</p>`,
                    // });
                    // if (!emailSent) {
                    //   return sendMessageResponse(
                    //     req,
                    //     res,
                    //     StatusCode.InternalServerError,
                    //     "Error sending verification email"
                    //   );
                    // }
                    return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.signIn.otpSent, Object.assign({ otp }, existingAppUser._doc));
                }
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.BadRequest, api_response_1.messages.signUp.emailTaken);
            }
            // Create the new user
            const newUser = new users_entity_1.default({
                email,
                role: role,
                userType: utills_1.USER_TYPE.SUB_ADMIN,
                status: userInfo.status,
            });
            const savedUser = yield newUser.save();
            yield savedUser.populate("role", "roleName _id");
            let fullName = firstName + " " + lastName;
            // Save AdminUser
            const adminUser = new admin_user_entity_1.default({
                userId: savedUser._id,
                firstName,
                lastName,
                fullName,
                phone,
                email,
                profileImage: (_a = req === null || req === void 0 ? void 0 : req.file) === null || _a === void 0 ? void 0 : _a.filename,
                role,
                userType: utills_1.USER_TYPE.SUB_ADMIN,
                status,
            });
            yield adminUser.save();
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Create, api_response_1.messages.user.success, savedUser);
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while add-user: ${error}`);
            console.error("Getting error while add-user:", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//==================== GET USERS =================/
function getUsers(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const { search, page, take, role } = req.query;
            const userInfo = (0, common_service_1.getTokenUserInfo)(req);
            const currentPage = parseInt(page) || 1;
            const perPage = parseInt(take) || 10;
            // Base aggregation query
            const matchQuery = {
                // fullName: { $nin: ["SUPER_ADMIN", userInfo?.fullName] },
                isDeleted: false,
            };
            // Apply search filtering
            if (search) {
                matchQuery.$and = [
                    // { fullName: { $nin: ["SUPER_ADMIN", userInfo?.fullName] } },
                    { fullName: new RegExp(search, "i") },
                ];
            }
            // Apply role filtering
            if (role) {
                matchQuery.role = role;
            }
            // Build the aggregation pipeline
            const pipeline = [
                { $match: matchQuery },
                {
                    $lookup: {
                        from: "roles", // Assuming 'roles' is the collection name for roles
                        localField: "role",
                        foreignField: "_id",
                        as: "role",
                    },
                },
                { $unwind: "$role" },
                { $skip: (currentPage - 1) * perPage },
                { $limit: perPage },
                {
                    $project: {
                        _id: 1,
                        profileImage: 1,
                        userId: 1,
                        email: 1,
                        firstName: 1,
                        lastName: 1,
                        fullName: 1,
                        phone: 1,
                        status: 1,
                        "role._id": 1,
                        "role.roleName": 1,
                        userType: 1,
                        __v: 1,
                    },
                },
            ];
            // Fetch data using aggregation
            const users = yield admin_user_entity_1.default.aggregate(pipeline);
            // Get total count for pagination
            const totalData = yield admin_user_entity_1.default.countDocuments(matchQuery);
            const totalPages = Math.ceil(totalData / perPage);
            if (!users.length) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.noDataAvailable);
            }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.success, {
                data: users,
                pagination: {
                    nextPage: currentPage < totalPages ? currentPage + 1 : null,
                    page: currentPage,
                    prevPage: currentPage > 1 ? currentPage - 1 : null,
                    take: perPage,
                    totalData,
                    totalPages,
                },
            });
        }
        catch (error) {
            console.error("Getting error while fetching users:", error);
            logger_service_1.logger.error(`Error while fetching users: ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//==================== GET USER =================/
function getUser(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            let user = yield admin_user_entity_1.default
                .findOne({
                _id: new mongoose_1.default.Types.ObjectId(req.params.id),
                isDeleted: false,
            })
                .populate("role", "roleName _id");
            if (!user) {
                user = yield app_user_entity_1.default.findOne({
                    _id: new mongoose_1.default.Types.ObjectId(req.params.id),
                    isDeleted: false,
                });
            }
            if (!user) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.noUsersFound);
            }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.success, user);
        }
        catch (error) {
            console.error("Getting error while fetching user : ", error);
            logger_service_1.logger.error(`Getting error while fetching user : ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//==================== DELETE USER =================/
function deleteUser(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const userId = req.params.id;
            // Attempt to soft delete the user in the adminRepository
            let user = yield admin_user_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(userId), isDeleted: false }, { $set: { isDeleted: true } }, { new: true });
            // If not found in adminRepository, attempt in appRepository
            if (!user) {
                user = yield app_user_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(userId), isDeleted: false }, { $set: { isDeleted: true } }, { new: true });
            }
            // If user is not found in both repositories, return NotFound response
            if (!user) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.noUsersFound);
            }
            // Get the main user ID and attempt to soft delete in userRepository
            const mainUserId = user.userId;
            const mainUser = yield users_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(mainUserId), isDeleted: false }, { $set: { isDeleted: true } }, { new: true });
            // If the main user is not found, return NotFound response
            if (!mainUser) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.noUsersFound);
            }
            // Return success response
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.delete);
        }
        catch (error) {
            console.error("Getting error while deleting user :", error);
            logger_service_1.logger.error(`Getting error while deleting user : ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//==================== UPDATE USER =================/
function updateUser(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const { firstName, lastName, email, password, _id, role, status, phone, industry, yourRoles, } = req.body;
        const profileImage = req.file ? req.file.filename : null;
        try {
            let fullName = firstName + " " + lastName;
            const updateData = {
                firstName,
                lastName,
                fullName,
                email,
                password,
                role,
                status,
                phone,
                industry,
                yourRoles,
            };
            if (profileImage) {
                updateData.profileImage = profileImage;
            }
            let updatedUser;
            let mainUserId;
            // Attempt to update the user in AdminUser collection
            updatedUser = yield admin_user_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(_id), isDeleted: false }, { $set: updateData }, { new: true });
            if (!updatedUser) {
                // If user is not found in AdminUser, try updating in AppUser collection
                updatedUser = yield app_user_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(_id), isDeleted: false }, { $set: updateData }, { new: true });
            }
            if (!updatedUser) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.updateUser.userNotFound);
            }
            mainUserId = updatedUser.userId;
            // Update the user in the User collection
            const updatedMainUser = yield users_entity_1.default.findOneAndUpdate({ _id: new mongoose_1.default.Types.ObjectId(mainUserId), isDeleted: false }, { $set: updateData }, { new: true });
            if (!updatedMainUser) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.NotFound, api_response_1.messages.updateUser.userNotFound);
            }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.updateUser.success, Object.assign(Object.assign({}, updatedUser._doc), { role: updatedMainUser.role, userType: updatedMainUser.userType }));
        }
        catch (error) {
            logger_service_1.logger.error(`Getting error while updating user : ${error}`);
            console.error("Getting error while updating user :", error);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
//==================== GET APP USER USER =================/
function getAppUsers(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const { search, page, take, role, isDropDown } = req.query;
            let userInfo = (0, common_service_1.getTokenUserInfo)(req);
            let searchQuery = { _id: { $ne: userInfo._id } };
            if (search) {
                searchQuery = {
                    $and: [
                        { _id: { $ne: new mongoose_1.default.Types.ObjectId(userInfo._id) } },
                        {
                            $or: [
                                { fullName: new RegExp(search, "i") },
                                { email: new RegExp(search, "i") },
                            ],
                        },
                    ],
                };
            }
            let filters = {};
            if (role)
                filters.role = role;
            let options;
            if (isDropDown === "true") {
                options = {
                    searchQuery,
                    filter: filters,
                };
            }
            else {
                options = {
                    searchQuery,
                    filter: filters,
                    page: parseInt(page) || 1,
                    take: parseInt(take) || 10,
                };
            }
            const { data: users, totalData, page: currentPage, take: perPage, totalPages, prevPage, nextPage, } = yield (0, common_service_1.fetchUsersWithFilterAndPagination)(app_user_entity_1.default, options);
            const populatedUsers = yield app_user_entity_1.default.populate(users, {
                path: "role",
                select: "roleName",
            });
            if (!users.length) {
                return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.noDataAvailable);
            }
            const data = {
                data: populatedUsers || [],
            };
            if (isDropDown !== "true") {
                data.pagination = {
                    nextPage,
                    page: currentPage,
                    prevPage,
                    take: perPage,
                    totalData,
                    totalPages,
                };
            }
            return (0, common_service_1.sendDataResponse)(req, res, api_response_1.StatusCode.Ok, api_response_1.messages.success, data);
        }
        catch (error) {
            console.error("Getting error while fetching users :", error);
            logger_service_1.logger.error(`Getting error while fetching users : ${error}`);
            return (0, common_service_1.sendMessageResponse)(req, res, api_response_1.StatusCode.InternalServerError, api_response_1.messages.internalServerError);
        }
    });
}
